# Copyright (c) 2009 ActiveState Software Inc.
# See http://www.activestate.com/activepython/license/ for licensing
# information.

from pypm.client import faq


class PyPMClientError(Exception): pass
class VersionConflict(PyPMClientError): pass
class NoPackage(PyPMClientError): pass


class NoPackageAvailable(NoPackage):
    """No such package available in the repositories"""

    def __init__(self, name, version_spec=None, while_doing=None):
        fullname = name
        if version_spec:
            fullname = '{0} ({1})'.format(fullname , version_spec)

        msg = "{0}Couldn't find package '{1}'. {2}".format(
            while_doing + ' Reason: ' if while_doing else '',
            fullname,
            faq.PACKAGE_NOT_FOUND.format(fullname))
        
        super(NoPackageAvailable, self).__init__(msg)


class NoPackageInstalled(NoPackage):
    """No such package installed"""

    def __init__(self, name, version):
        msg = '{0}{1} is not installed'.format(
            name,
            '-'+version if version else '')
        super(NoPackageInstalled, self).__init__(msg)


class DependencyNotFound(NoPackageAvailable):

    def __init__(self, for_package_name, requirement):
        while_doing = 'Cannot satisfy requirement "{0}" for {1}.'.format(
            requirement, for_package_name)
        super(DependencyNotFound, self).__init__(
            requirement.project_name,
            requirement,
            while_doing)


class PackageAccessError(PyPMClientError):
    """Errors related to accessing a package from ActiveState repository
    
    Eg: If a Business Edition subscription is expired, one cannot install
    the BE package.
    """
    
    def __init__(self, p, reason, msg):
        """
        - p:       The package that failed to install
        - reason:  The actual reason for the failure
        - msg:     The diagnostic message we show to the user; separated by \n
        """
        self.p = p
        self.reason = reason
        self.msg = msg
        
    def __str__(self):
        tmpl = r'''Can't install {self.p.full_name}: {self.reason}

{starred_msg}'''
        return tmpl.format(
            self=self,
            starred_msg = '*** ' + ('\n*** '.join(self.msg.splitlines())))
    