# Copyright (c) 2009 ActiveState Software Inc.
# See http://www.activestate.com/activepython/license/ for licensing
# information.

"""
    pypm.client.app
    ~~~~~~~~~~~~~~~

    Generic application routines
"""

import sys
from os.path import join, exists, expanduser
import tempfile

from pypm.common.util import lazyproperty, get_windows_local_appdata_path


class Application(object):
    """Represents the PyPM client application"""
    name = 'PyPM'
    company = 'ActiveState'

    @lazyproperty
    def log_file_location(self):
        """Preferred log file location"""
        if sys.platform.startswith('win'):
            return join(self.appdata_location, self.name + '.log')
        elif sys.platform.startswith('darwin'): # MacOSX
            return join(expanduser('~/Library/Logs'), self.name+'.log')
        else: # Unix
            return join(self.appdata_location, self.name.lower()+'.log')

    @lazyproperty
    def appdata_location(self):
        """Location of directory where application support files are stored."""
        if sys.platform.startswith('win'):
            return join(get_windows_local_appdata_path(),
                        self.company, self.name)
        elif sys.platform.startswith('darwin'):
            return join(expanduser('~/Library/Application Support/'), self.name)
        else:
            return join(expanduser('~'), '.'+self.name.lower())

    @lazyproperty
    def cache_location(self):
        """Location of directory where cache files can be stored"""
        if sys.platform.startswith('win'):
            return join(tempfile.gettempdir(), self.name)
        elif sys.platform.startswith('darwin'):
            return join(expanduser('~/Library/Caches/'), self.name)
        else:
            # no user-specific tmp/cache directory on unices
            return self.appdata_location
            
application = Application()
